/************************************************************************************************************\

Module Name:    AoutAlsaModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AoutAlsaModule.h"
#include "CommonUtils.h"

#ifdef ALSA_SUPPORT
#include "alsa/asoundlib.h"
#endif

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------
#ifdef ALSA_SUPPORT

#define MAX_ALSA_DEVICES 32
static LSTR64  g_aszDevNames[MAX_ALSA_DEVICES] = {0};
static MUINT32 g_uiNbDev = 0;

/************************************************************************************************************\

Function:       AoutAlsaMod_EnumDevices

Description:    .

\************************************************************************************************************/
LStatus AoutAlsaMod_EnumDevices(MUINT32* puiNbDevices)
{
    LStatus eStatus = EnumAlsaDeviceNames(
                                g_aszDevNames,
                                MAX_ALSA_DEVICES,
                                &g_uiNbDev,
                                MFALSE);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        *puiNbDevices = g_uiNbDev;
    }

    return eStatus;
}


/************************************************************************************************************\

Function:       AoutAlsaMod_GetDeviceName

Description:    .

\************************************************************************************************************/
LStatus AoutAlsaMod_GetDeviceName(MUINT32 uiDeviceIdx, LSTR64* pszName)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    if ((uiDeviceIdx < g_uiNbDev)
        && (pszName != MNULL))
    {
        strcpy(*pszName, g_aszDevNames[uiDeviceIdx]);
        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Init

Description:    .

\************************************************************************************************************/
LStatus AoutAlsaMod_Init(AoutAlsaModule* poAout,
    MUINT32 uiDevIdx,
    MUINT32 uiSamplingRate,
    MUINT32 uiNbChannels,
    MUINT32 uiSampleSize,
    MBOOL32 bBigEndian,
    MBOOL32 bUnsigned,
    MUINT32 uiFramesPerBuffer)
{
    MsgLog(2, "{...");

    AoutAlsaMod_Cleanup(poAout);

    LStatus eStatus = LStatus_INVALID_PARAM;

    if((poAout != MNULL)
        && (uiDevIdx < g_uiNbDev)
        && (uiFramesPerBuffer > 0))
    {
        poAout->uiSamplingRate      = uiSamplingRate;
        poAout->uiNbChannels        = uiNbChannels;
        poAout->uiFramesPerBuffer   = uiFramesPerBuffer;
        poAout->uiSampleSize        = uiSampleSize;
        poAout->bBigEndian          = bBigEndian;
        poAout->bUnsigned           = bUnsigned;
        poAout->uiDevIdx            = uiDevIdx;
        eStatus = LStatus_OK;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        int iErr = 0;
        if((iErr = snd_pcm_open(
                        (snd_pcm_t**)&poAout->hAlsaDev,
                        g_aszDevNames[uiDevIdx],
                        SND_PCM_STREAM_PLAYBACK,
                        0)) < 0)
        {
            MsgLogErr("Error, snd_pcm_open(%s) failed: %s",
                      g_aszDevNames[uiDevIdx], snd_strerror(iErr));
            eStatus = LStatus_FAIL;
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ConfigureAlsaDevice(
                    poAout->hAlsaDev,
                    poAout->uiSamplingRate,
                    poAout->uiNbChannels,
                    poAout->uiSampleSize,
                    poAout->bBigEndian,
                    poAout->bUnsigned,
                    poAout->uiFramesPerBuffer);

        if(LSTATUS_IS_FAIL(eStatus))
        {
            MsgLogErr("Error, can't configure ALSA device");
        }
    }

    if (LSTATUS_IS_FAIL(eStatus))
    {
        AoutAlsaMod_Cleanup(poAout);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Cleanup

Description:    .

\************************************************************************************************************/
void AoutAlsaMod_Cleanup(AoutAlsaModule* poAout)
{
    if (poAout != MNULL)
    {
        if (poAout->hAlsaDev != MNULL)
        {
            snd_pcm_close(poAout->hAlsaDev);
            poAout->hAlsaDev = MNULL;
        }
    }
}

/************************************************************************************************************\

Function:       AoutAlsaMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus AoutAlsaMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AoutAlsaModule* poAout = (AoutAlsaModule*)pvData;

    ModThread_SetName(g_aszDevNames[poAout->uiDevIdx]);
    MsgLog(2, "Start thread %p.", pthread_self());

    MUINT32 uiErrorCount = 0;

    while (!poAout->oCpuThread.bKillThread)
    {
        BufferInfo* poSrcBuffer = MNULL;

        LStatus eStatus = ModLnkIn_GetSubmittedBuffer(
                                &(poAout->oInLink),
                                100,
                                0,
                                MNULL,
                                &poSrcBuffer,
                                MNULL,
                                MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poAout->oCpuThread.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else
            {
                MUINT8* puiBuffer = MNULL;
                eStatus = LBuffer_BeginAccess(poSrcBuffer->hBuffer, 0, 1, &puiBuffer);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    snd_pcm_state_t eState = snd_pcm_state(poAout->hAlsaDev);

                    if((eState != SND_PCM_STATE_PREPARED)
                        && (eState != SND_PCM_STATE_RUNNING))
                    {
                        uiErrorCount++;
                        MsgLog(1, "Warning, state=%u, preparing PCM device. (error count=%u)", eState, uiErrorCount);
                        snd_pcm_prepare(poAout->hAlsaDev);
                    }

                    snd_pcm_sframes_t iInputFrames = snd_pcm_bytes_to_frames(
                                                         poAout->hAlsaDev,
                                                         poSrcBuffer->uiSizeBytes);

                    snd_pcm_sframes_t iWrittenFrames = snd_pcm_writei(
                                                        poAout->hAlsaDev,
                                                        puiBuffer + poSrcBuffer->uiStartOffset,
                                                        iInputFrames);

                    MsgLog(4, "snd_pcm_writei: written %d frames out of %d",
                           iWrittenFrames, iInputFrames);

                    if(iWrittenFrames >= 0)
                    {
                        if(iWrittenFrames != iInputFrames)
                        {
                            uiErrorCount++;
                            MsgLog(1, "Warning, written %u frames out of %u (error count=%u)",
                                   iWrittenFrames, iInputFrames, uiErrorCount);
                        }
                    }
                    else
                    {
                        uiErrorCount++;
                        MsgLog(0, "Error in snd_pcm_writei: %s (error count=%u)",
                               snd_strerror(iWrittenFrames), uiErrorCount);
                    }

                    LBuffer_EndAccess(poSrcBuffer->hBuffer);
                }
            }

            ModLnkIn_ReturnBuffer(&(poAout->oInLink), poSrcBuffer, MNULL, NO_TAG);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}


/************************************************************************************************************\

Function:       AoutAlsaMod_Start

Description:    .

\************************************************************************************************************/
LStatus AoutAlsaMod_Start(AoutAlsaModule* poAout)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poAout != MNULL)
        && (poAout->hAlsaDev != MNULL))
    {
        if (poAout->oCpuThread.hThread == MNULL)
        {
            eStatus = ModThread_Start(&(poAout->oCpuThread), poAout, AoutAlsaMod_CpuThread);
        }
        else
        {
            eStatus = LStatus_OK;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Stop

Description:    .

\************************************************************************************************************/
void AoutAlsaMod_Stop(AoutAlsaModule* poAout)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if (poAout != MNULL)
    {
        eStatus = ModThread_Stop(&poAout->oCpuThread);
        if(eStatus != LStatus_OK)
        {
            MsgLog(0, "ModThread_Stop failed (%d)\n", eStatus);
        }
    }

    MsgLog(2, "...}");
}

#else

/************************************************************************************************************\

Function:       AoutAlsaMod_EnumDevices

Description:    Stub.

\************************************************************************************************************/
LStatus AoutAlsaMod_EnumDevices(MUINT32* puiNbDevices)
{
    (void)puiNbDevices;
    MsgLog(1, "No ALSA support.");
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_GetDeviceName

Description:    Stub.

\************************************************************************************************************/

LStatus AoutAlsaMod_GetDeviceName(MUINT32 uiDeviceIdx, LSTR64* pszName)
{
    (void)uiDeviceIdx;
    (void)pszName;
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Init

Description:    Stub.

\************************************************************************************************************/
LStatus AoutAlsaMod_Init(AoutAlsaModule* poAout,
    MUINT32 uiDevIdx,
    MUINT32 uiSamplingRate,
    MUINT32 uiNbChannels,
    MUINT32 uiSampleSize,
    MBOOL32 bBigEndian,
    MBOOL32 bUnsigned,
    MUINT32 uiFramesPerBuffer)
{
    (void)poAout;
    (void)uiDevIdx;
    (void)uiSamplingRate;
    (void)uiNbChannels;
    (void)uiSampleSize;
    (void)bBigEndian;
    (void)bUnsigned;
    (void)uiFramesPerBuffer;
    MsgLog(1, "No ALSA support.");
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Cleanup

Description:    Stub.

\************************************************************************************************************/
void AoutAlsaMod_Cleanup(AoutAlsaModule* poAout)
{
    (void)poAout;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Start

Description:    Stub.

\************************************************************************************************************/
LStatus AoutAlsaMod_Start(AoutAlsaModule* poAout)
{
    (void)poAout;
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AoutAlsaMod_Stop

Description:    Stub.

\************************************************************************************************************/
void AoutAlsaMod_Stop(AoutAlsaModule* poAout)
{
    (void)poAout;
}

#endif
